/*************************************************************************
 * ri.h - header file for the Blue Moon Rendering Tools (BMRT)           *
 *************************************************************************/

#ifndef RI_H
#define RI_H 1

#ifdef __cplusplus
extern "C" {
#endif

typedef short   RtBoolean;
typedef int     RtInt;
typedef float   RtFloat;

typedef char   *RtToken;

#ifndef NCOMPS
#define NCOMPS 3
#endif

typedef RtFloat RtColor[NCOMPS];
typedef RtFloat RtPoint[3];
typedef RtFloat RtVector[3];
typedef RtFloat RtNormal[3];
typedef RtFloat RtHpoint[3];
typedef RtFloat RtMatrix[4][4];
typedef RtFloat RtBasis[4][4];
typedef RtFloat RtBound[6];
typedef char   *RtString;

typedef char   *RtPointer;
#define RtVoid void
typedef RtFloat (*RtFilterFunc)(RtFloat, RtFloat, RtFloat, RtFloat);
typedef RtVoid  (*RtErrorHandler)(RtInt code, RtInt severity, char *msg);
typedef RtVoid  (*RtProcSubdivFunc)(RtPointer, RtFloat);
typedef RtVoid  (*RtProcFreeFunc)(RtPointer);
typedef RtVoid  (*RtArchiveCallback)(RtToken, char *, ...);

typedef RtVoid  (*RtFunc)();

typedef RtPointer RtObjectHandle;
typedef RtPointer RtLightHandle;
typedef RtPointer RtContextHandle;

#define RI_FALSE    0
#define RI_TRUE     1
#define RI_INFINITY (RtFloat)1.0e38
#define RI_EPSILON  (RtFloat)1.0e-10

#ifndef NULL
#define NULL 0
#endif
#define RI_NULL NULL


extern RtToken  RI_FRAMEBUFFER, RI_FILE;
extern RtToken  RI_RGB, RI_RGBA, RI_RGBZ, RI_RGBAZ, RI_A, RI_Z, RI_AZ;
extern RtToken  RI_PERSPECTIVE, RI_ORTHOGRAPHIC;
extern RtToken  RI_HIDDEN, RI_PAINT;
extern RtToken  RI_CONSTANT, RI_SMOOTH;
extern RtToken  RI_FLATNESS, RI_FOV;

extern RtToken  RI_AMBIENTLIGHT, RI_POINTLIGHT, RI_DISTANTLIGHT, RI_SPOTLIGHT;
extern RtToken  RI_INTENSITY, RI_LIGHTCOLOR, RI_FROM, RI_TO, RI_CONEANGLE,
                RI_CONEDELTAANGLE, RI_BEAMDISTRIBUTION;
extern RtToken  RI_MATTE, RI_METAL, RI_SHINYMETAL,
                RI_PLASTIC, RI_PAINTEDPLASTIC;
extern RtToken  RI_KA, RI_KD, RI_KS, RI_ROUGHNESS, RI_KR,
                RI_TEXTURENAME, RI_SPECULARCOLOR;
extern RtToken  RI_DEPTHCUE, RI_FOG, RI_BUMPY;
extern RtToken  RI_MINDISTANCE, RI_MAXDISTANCE, RI_BACKGROUND,
                RI_DISTANCE, RI_AMPLITUDE;

extern RtToken  RI_INSIDE, RI_OUTSIDE, RI_LH, RI_RH;
extern RtToken  RI_P, RI_PZ, RI_PW, RI_N, RI_NP, RI_CS,
                RI_OS, RI_S, RI_T, RI_ST;
extern RtToken  RI_BILINEAR, RI_BICUBIC;
extern RtToken  RI_PRIMITIVE, RI_INTERSECTION, RI_UNION, RI_DIFFERENCE;
extern RtToken  RI_PERIODIC, RI_NONPERIODIC, RI_CLAMP, RI_BLACK;
extern RtToken  RI_IGNORE, RI_PRINT, RI_ABORT, RI_HANDLER;
extern RtToken  RI_ORIGIN, RI_IDENTIFIER, RI_NAME;
extern RtToken  RI_COMMENT, RI_STRUCTURE, RI_VERBATIM;
extern RtToken  RI_LINEAR, RI_CUBIC, RI_WIDTH, RI_CONSTANTWIDTH;

extern RtToken  RI_CURRENT, RI_WORLD, RI_OBJECT, RI_SHADER;
extern RtToken  RI_RASTER, RI_NDC, RI_SCREEN, RI_CAMERA, RI_EYE;

extern RtToken  RI_HOLE, RI_CREASE, RI_CORNER, RI_INTERPOLATEBOUNDARY;

extern RtBasis  RiBezierBasis, RiBSplineBasis, RiCatmullRomBasis,
                RiHermiteBasis, RiPowerBasis;

#define RI_BEZIERSTEP       ((RtInt)3)
#define RI_BSPLINESTEP      ((RtInt)1)
#define RI_CATMULLROMSTEP   ((RtInt)1)
#define RI_HERMITESTEP      ((RtInt)2)
#define RI_POWERSTEP        ((RtInt)4)

extern RtInt RiLastError;



/* Basic control flow, scoping, stacks */
RtVoid RiBegin (RtToken name);
RtVoid RiEnd (void);
RtContextHandle RiGetContext (void);
RtVoid RiContext (RtContextHandle handle);
RtVoid RiFrameBegin (RtInt number);
RtVoid RiFrameEnd (void);
RtVoid RiWorldBegin (void);
RtVoid RiWorldEnd (void);
RtVoid RiAttributeBegin (void);
RtVoid RiAttributeEnd (void);
RtVoid RiTransformBegin (void);
RtVoid RiTransformEnd (void);
RtVoid RiSolidBegin (RtToken type);
RtVoid RiSolidEnd(void);
RtVoid RiMotionBegin (RtInt N, ...);
RtVoid RiMotionBeginV (RtInt N, RtFloat times[]);
RtVoid RiMotionEnd (void);
RtToken RiDeclare (char *name, char *declaration);

/* Options */
RtVoid RiClipping (RtFloat hither, RtFloat yon);
RtVoid RiColorSamples (RtInt N, RtFloat *nRGB, RtFloat *RGBn);
RtVoid RiCropWindow (RtFloat xmin, RtFloat xmax, RtFloat ymin, RtFloat ymax);
RtVoid RiDepthOfField (RtFloat fstop, RtFloat focallength, RtFloat focaldistance);
RtVoid RiDisplay (char *name, RtToken type, RtToken mode, ...);
RtVoid RiDisplayV (char *name, RtToken type, RtToken mode,
		   RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiExposure (RtFloat gain, RtFloat gamma);
RtVoid RiFormat (RtInt xres, RtInt yres, RtFloat aspect);
RtVoid RiFrameAspectRatio (RtFloat aspect);
RtVoid RiHider (RtToken type, ...);
RtVoid RiHiderV (RtToken type, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiImager (char *name, ...);
RtVoid RiImagerV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiOption (char *name, ...);
RtVoid RiOptionV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPixelFilter (RtFilterFunc function, RtFloat xwidth, RtFloat ywidth);
RtVoid RiPixelSamples (RtFloat xsamples, RtFloat ysamples);
RtVoid RiPixelVariance (RtFloat variation);
RtVoid RiProjection (char *name, ...);
RtVoid RiProjectionV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiQuantize (RtToken type, RtInt one, RtInt qmin, RtInt qmax, RtFloat ampl);
RtVoid RiRelativeDetail (RtFloat relativedetail);
RtVoid RiScreenWindow (RtFloat left, RtFloat right, RtFloat bot, RtFloat top);
RtVoid RiShutter (RtFloat smin, RtFloat smax);


/* Filters */
RtFloat RiBoxFilter (RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth);
RtFloat RiCatmullRomFilter (RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth);
RtFloat RiGaussianFilter (RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth);
RtFloat RiSincFilter (RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth);
RtFloat RiTriangleFilter (RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth);

/* Attributes */
RtVoid RiAttribute (char *name, ...);
RtVoid RiAttributeV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiBound (RtBound bound);
RtVoid RiColor (RtColor Cs);
RtVoid RiDetail (RtBound bound);
RtVoid RiDetailRange (RtFloat minvis, RtFloat lowtran,
		      RtFloat uptran, RtFloat maxvis);
RtVoid RiGeometricApproximation (RtToken type, RtFloat value);
RtVoid RiGeometricRepresentation (RtToken type);
RtVoid RiIlluminate (RtLightHandle light, RtBoolean onoff);
RtVoid RiMatte (RtBoolean onoff);
RtVoid RiOpacity (RtColor Cs);
RtVoid RiOrientation (RtToken orientation);
RtVoid RiReverseOrientation (void);
RtVoid RiShadingInterpolation (RtToken type);
RtVoid RiShadingRate (RtFloat size);
RtVoid RiSides (RtInt nsides);
RtVoid RiTextureCoordinates (RtFloat s1, RtFloat t1, RtFloat s2, RtFloat t2,
			     RtFloat s3, RtFloat t3, RtFloat s4, RtFloat t4);

/* Shaders */
RtLightHandle RiAreaLightSource (char *name, ...);
RtLightHandle RiAreaLightSourceV (char *name, RtInt n,
				  RtToken tokens[], RtPointer params[]);
RtVoid RiAtmosphere (char *name, ...);
RtVoid RiAtmosphereV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiDisplacement (char *name, ...);
RtVoid RiDisplacementV (char *name, RtInt n,
			RtToken tokens[], RtPointer params[]);
RtVoid RiExterior (char *name, ...);
RtVoid RiExteriorV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiInterior (char *name, ...);
RtVoid RiInteriorV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);
RtLightHandle RiLightSource (char *name, ...);
RtLightHandle RiLightSourceV (char *name, RtInt n,
			      RtToken tokens[], RtPointer params[]);
RtVoid RiSurface (char *name, ...);
RtVoid RiSurfaceV (char *name, RtInt n, RtToken tokens[], RtPointer params[]);

/* Transformations */
RtVoid RiConcatTransform (RtMatrix transform);
RtVoid RiCoordinateSystem (RtToken space);
RtVoid RiCoordSysTransform (RtToken space);
RtVoid RiIdentity (void);
RtVoid RiPerspective (RtFloat fov);
RtVoid RiRotate (RtFloat angle, RtFloat dx, RtFloat dy, RtFloat dz);
RtVoid RiScale (RtFloat dx, RtFloat dy, RtFloat dz);
RtVoid RiSkew (RtFloat angle, RtFloat dx1, RtFloat dy1, RtFloat dz1,
	       RtFloat dx2, RtFloat dy2, RtFloat dz2);
RtVoid RiTransform (RtMatrix transform);
RtVoid RiTranslate (RtFloat dx, RtFloat dy, RtFloat dz);

RtPoint *RiTransformPoints (RtToken fromspace, RtToken tospace,
			    RtInt npoints, RtPoint *points);

/* Geometric Primitives (and a couple gprim-specific attributes) */
RtVoid RiBasis (RtBasis ubasis, RtInt ustep, RtBasis vbasis, RtInt vstep);
RtVoid RiBlobby (RtInt nleaf, RtInt ncode, RtInt code[],
		 RtInt nflt, RtFloat flt[], RtInt nstr, RtString str[], ...);
RtVoid RiBlobbyV (RtInt nleaf, RtInt ncode, RtInt code[],
		  RtInt nflt, RtFloat flt[], RtInt nstr, RtString str[],
		  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiCone (RtFloat height, RtFloat radius, RtFloat thetamax, ...);
RtVoid RiConeV (RtFloat height, RtFloat radius, RtFloat thetamax, 
		RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiCurves (RtToken degree, RtInt ncurves,
		 RtInt nverts[], RtToken wrap, ...);
RtVoid RiCurvesV (RtToken degree, RtInt ncurves, RtInt nverts[], RtToken wrap,
		  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiCylinder (RtFloat radius, RtFloat zmin, RtFloat zmax, 
		   RtFloat thetamax, ...);
RtVoid RiCylinderV (RtFloat radius, RtFloat zmin, RtFloat zmax, RtFloat thetamax,
		    RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiDisk (RtFloat height, RtFloat radius, RtFloat thetamax, ...);
RtVoid RiDiskV (RtFloat height, RtFloat radius, RtFloat thetamax, 
		RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiGeneralPolygon (RtInt nloops, RtInt *nverts, ...);
RtVoid RiGeneralPolygonV (RtInt nloops, RtInt *nverts,
			  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiGeometry (RtToken type, ...);
RtVoid RiGeometryV (RtToken type, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiHyperboloid (RtPoint point1, RtPoint point2, RtFloat thetamax, ...);
RtVoid RiHyperboloidV (RtPoint point1, RtPoint point2, RtFloat thetamax, 
		       RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiNuPatch (RtInt nu, RtInt uorder, RtFloat *uknot,
		  RtFloat umin, RtFloat umax, RtInt nv, RtInt vorder,
		  RtFloat *vknot, RtFloat vmin, RtFloat vmax, ...);
RtVoid RiNuPatchV (RtInt nu, RtInt uorder, RtFloat *uknot,
		   RtFloat umin, RtFloat umax, RtInt nv, RtInt vorder,
		   RtFloat *vknot, RtFloat vmin, RtFloat vmax, 
		   RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiParaboloid (RtFloat rmax, RtFloat zmin,
		     RtFloat zmax, RtFloat thetamax, ...);
RtVoid RiParaboloidV (RtFloat rmax, RtFloat zmin, RtFloat zmax, RtFloat thetamax, 
		      RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPatch (RtToken type, ...);
RtVoid RiPatchV (RtToken type, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPatchMesh (RtToken type, RtInt nu, RtToken uwrap,
		    RtInt nv, RtToken vwrap, ...);
RtVoid RiPatchMeshV (RtToken type, RtInt nu, RtToken uwrap, RtInt nv,
		     RtToken vwrap, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPoints (RtInt npts,...);
RtVoid RiPointsV (RtInt npts, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPointsGeneralPolygons (RtInt npolys, RtInt *nloops,
				RtInt *nverts, RtInt *verts, ...);
RtVoid RiPointsGeneralPolygonsV (RtInt npolys, RtInt *nloops,
				 RtInt *nverts, RtInt *verts, 
				 RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPointsPolygons (RtInt npolys, RtInt *nverts, RtInt *verts, ...);
RtVoid RiPointsPolygonsV (RtInt npolys, RtInt *nverts, RtInt *verts, 
			  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiPolygon (RtInt nvertices, ...);
RtVoid RiPolygonV (RtInt nvertices, RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiSphere (RtFloat radius, RtFloat zmin,
		 RtFloat zmax, RtFloat thetamax, ...);
RtVoid RiSphereV (RtFloat radius, RtFloat zmin, RtFloat zmax, RtFloat thetamax,
		  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiSubdivisionMesh (RtToken scheme, RtInt nfaces,
			  RtInt nvertices[], RtInt vertices[],
			  RtInt ntags, RtToken tags[], RtInt nargs[],
			  RtInt intargs[], RtFloat floatargs[], ...);
RtVoid RiSubdivisionMeshV (RtToken scheme, RtInt nfaces,
			   RtInt nvertices[], RtInt vertices[],
			   RtInt ntags, RtToken tags[], RtInt nargs[],
			   RtInt intargs[], RtFloat floatargs[], 
			   RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiTorus (RtFloat majorrad, RtFloat minorrad, RtFloat phimin,
		RtFloat phimax, RtFloat thetamax, ...);
RtVoid RiTorusV (RtFloat majorrad, RtFloat minorrad, RtFloat phimin,
		 RtFloat phimax, RtFloat thetamax,
		 RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiTrimCurve (RtInt nloops, RtInt *ncurves, RtInt *order,
		    RtFloat *knot, RtFloat *amin, RtFloat *amax,
		    RtInt *n, RtFloat *u, RtFloat *v, RtFloat *w);
/* Procedural primitives */
RtVoid RiProcedural (RtPointer data, RtBound bound,
		     RtVoid (*subdivfunc) (RtPointer, RtFloat),
		     RtVoid (*freefunc) (RtPointer));
RtVoid RiProcDelayedReadArchive (RtPointer data, RtFloat detail);
RtVoid RiProcRunProgram (RtPointer data, RtFloat detail);
RtVoid RiProcDynamicLoad (RtPointer data, RtFloat detail);
/* Object Instancing */
RtObjectHandle RiObjectBegin (void);
RtVoid RiObjectEnd (void);
RtVoid RiObjectInstance (RtObjectHandle handle);



/* Texture map creation */
RtVoid RiMakeCubeFaceEnvironment (char *px, char *nx, char *py,
				  char *ny, char *pz, char *nz,
				  char *tex, RtFloat fov, RtFilterFunc filterfunc,
				  RtFloat swidth, RtFloat twidth, ...);
RtVoid RiMakeCubeFaceEnvironmentV (char *px, char *nx, char *py,
				   char *ny, char *pz, char *nz,
				   char *tex, RtFloat fov, RtFilterFunc filterfunc,
				   RtFloat swidth, RtFloat twidth,
				   RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiMakeLatLongEnvironment (char *pic, char *tex, RtFilterFunc filterfunc,
				 RtFloat swidth, RtFloat twidth, ...);
RtVoid RiMakeLatLongEnvironmentV (char *pic, char *tex, RtFilterFunc filterfunc,
				  RtFloat swidth, RtFloat twidth,
				  RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiMakeShadow (char *pic, char *tex, ...);
RtVoid RiMakeShadowV (char *pic, char *tex,
		      RtInt n, RtToken tokens[], RtPointer params[]);
RtVoid RiMakeTexture (char *pic, char *tex, RtToken swrap, RtToken twrap,
		      RtFilterFunc filterfunc, RtFloat swidth, RtFloat twidth, ...);
RtVoid RiMakeTextureV (char *pic, char *tex, RtToken swrap, RtToken twrap,
		       RtFilterFunc filterfunc, RtFloat swidth, RtFloat twidth,
		       RtInt n, RtToken tokens[], RtPointer params[]);

/* Error Handlers */
RtVoid RiErrorHandler (RtErrorHandler handler);
RtVoid RiErrorAbort (RtInt code, RtInt severity, char *message);
RtVoid RiErrorIgnore (RtInt code, RtInt severity, char *message);
RtVoid RiErrorPrint (RtInt code, RtInt severity, char *message);

/* Reading and writing archive files */
RtVoid RiArchiveRecord (RtToken type, char *format, ...);
RtVoid RiReadArchive (RtString filename, RtArchiveCallback callback, ...);
RtVoid RiReadArchiveV (RtString filename, RtArchiveCallback callback,
		       int n, RtToken tokens[], RtPointer params[]);



/*
    Error codes

     1 - 10     System and File errors
    11 - 20     Program Limitations
    21 - 40     State Errors
    41 - 60     Parameter and Protocol Errors
    61 - 80     Execution Errors
*/

#define RIE_NOERROR         0

#define RIE_NOMEM           1       /* Out of memory */
#define RIE_SYSTEM          2       /* Miscellaneous system error */
#define RIE_NOFILE          3       /* File nonexistant */
#define RIE_BADFILE         4       /* Bad file format */
#define RIE_VERSION         5       /* File version mismatch */
#define RIE_DISKFULL        6       /* Target disk is full */

#define RIE_INCAPABLE      11       /* Optional RI feature */
#define RIE_OPTIONAL       11       /* Optional RI feature */
#define RIE_UNIMPLEMENT    12       /* Unimplemented feature */
#define RIE_LIMIT          13       /* Arbitrary program limit */
#define RIE_BUG            14       /* Probably a bug in renderer */

#define RIE_NOTSTARTED     23       /* RiBegin not called */
#define RIE_NESTING        24       /* Bad begin-end nesting */
#define RIE_NOTOPTIONS     25       /* Invalid state for options */
#define RIE_NOTATTRIBS     26       /* Invalid state for attributes */
#define RIE_NOTPRIMS       27       /* Invalid state for primitives */
#define RIE_ILLSTATE       28       /* Other invalid state */
#define RIE_BADMOTION      29       /* Badly formed motion block */
#define RIE_BADSOLID       30       /* Badly formed solid block */

#define RIE_BADTOKEN       41       /* Invalid token for request */
#define RIE_RANGE          42       /* Parameter out of range */
#define RIE_CONSISTENCY    43       /* Parameters inconsistent */
#define RIE_BADHANDLE      44       /* Bad object/light handle */
#define RIE_NOSHADER       45       /* Can't load requested shader */
#define RIE_MISSINGDATA    46       /* Required parameters not provided */
#define RIE_SYNTAX         47       /* Declare type syntax error */

#define RIE_MATH           61       /* Zerodivide, noninvert matrix, etc. */


/* Error severity levels */

#define RIE_INFO            0       /* Rendering stats & other info */
#define RIE_WARNING         1       /* Something seems wrong, maybe okay */
#define RIE_ERROR           2       /* Problem.  Results may be wrong */
#define RIE_SEVERE          3       /* So bad you should probably abort */

#ifdef __cplusplus
}
#endif


#endif  /* RI_H */

